# SWAI Web App (.swai) File Guide

This document explains how to create a SWAI Web App by writing a `.swai` YAML file. The file describes your web app’s metadata, the main URL to load, and navigation/security behavior. SWAI reads this file to create an Electron-based window and enforce navigation rules.

## Field reference

Required
- app_name (string): Display name used for the window/titlebar.
- app_id (string): Unique identifier. Used for per-app data paths and window classing.
- main_url (string): Initial URL to load.
- allowed_urls (string[]): List of URL patterns the app is allowed to navigate to internally. Supports wildcards `*`. Outside URLs will redirect to a system browser.

Common optional
- not_allowed_urls (string[]): URL patterns that are explicitly disallowed (overrides allowed_urls). If a navigation matches here, SWAI will open it in the system browser instead of the app.
- icon_path/icon_url (string): Local path to an icon file. Alternatively, some test apps use icon_url. Either can be left empty.
- categories (string[]): FreeDesktop categories for desktop integration.
- keywords (string[]): Extra search keywords. Used for stillCenter.
- comment (string): Short description. Used for stillCenter.
- mimetypes (string[]): Declares mimetypes the app can handle (planned/optional). This does nothing yet, but will in the future.
- allow_multi_window (boolean, default false): If true, links that match allowed_urls can open in new app windows. If false, such links are opened in transient modal windows instead. This option also allows users to open windows of the same app.
- ecosystem (string): Groups related apps to share session/cookies and integrate with the SwaiRegistry (DBus). Windows may keep an internal title for WM_CLASS while displaying the live page title. This allows apps related to each other such as G Suite apps to open urls in each other while sharing a login, for example Google Drive opening Documents inside Google Docs.
- extensions (string[]): Reserved for future browser extension support.
- login_urls (string[]): URL patterns that are always allowed while currently on a matching login page. This is for allowing third-party logins to properly work.
- use_dynamic_titlebar_colors (boolean, default true): If true, SWAI will automatically detect and apply the website's theme color to the titlebar (similar to Safari's web app feature). Set to false to disable this feature and use the default system titlebar colors.
- use_manifest (boolean, default false): If true, SWAI will fetch the website's manifest.json and extend the app configuration with properties from it. Specifically, it will add URLs from `scope`, `scope_extensions`, and `start_url` to the allowed_urls list. This is useful for progressive web apps that define their navigation scope in the manifest.

Other notes parsed from the .swai file are stored but may be unused in the current codebase; they are kept for forward compatibility.

## Navigation behavior

SWAI enforces navigation rules at runtime:
- Will-navigate: If a navigation target does not pass _get_navigation_allowed, it is prevented and opened in the default system browser instead.
- Did-navigate and Did-navigate-in-page: The window titlebar is updated to reflect the current page title, and the titlebar receives a url-changed event to update back/forward availability.

Allowed logic (simplified):
1) If navigationUrl matches not_allowed_urls → Open in browser (blocked in app).
2) If currentUrl matches login_urls → Allow (to complete login flows).
3) If currentUrl is NOT within allowed_urls → Allow (initial navigations into the app).
4) If navigationUrl matches allowed_urls → Allow.
5) Otherwise → Open in browser.

Tip: Always include all internal routes in allowed_urls. For SPAs, wildcard patterns like `https://app.example.com/*` are recommended.

## Ecosystem behavior

If ecosystem is set, SWAI:
- Shares the session partition between all apps with the same ecosystem so cookies/auth persist across them.
- Registers an internal window title via SwaiRegistry, while keeping a live display title in the UI.

This enables seamless SSO and cohesive behavior across a suite (e.g., Microsoft 365 apps).

## Manifest integration

When `use_manifest: true` is set, SWAI will fetch the website's web app manifest (manifest.json) after the page loads and extend the app's configuration with the following properties:

- **scope**: Defines the navigation scope for the web app. SWAI adds this as a wildcard pattern to allowed_urls.
- **scope_extensions**: An array of additional origins that the web app can navigate to. Each origin is added to allowed_urls.
- **start_url**: The preferred URL to launch the app. Added to allowed_urls if not already present.

This is particularly useful for Progressive Web Apps (PWAs) that define their navigation boundaries in the manifest rather than requiring manual configuration.

Example:
```yaml
app_name: My PWA
app_id: com.example.mypwa
main_url: https://example.com/app/
allowed_urls:
  - "https://example.com/*"
use_manifest: true  # Will automatically add scope and scope_extensions from manifest
```

Note: The manifest properties **extend** the existing allowed_urls—they don't replace them. Your .swai file configuration takes precedence.

## Dynamic titlebar colors

SWAI automatically detects and applies the website's theme color to the titlebar, similar to Safari's web app feature. The detection order is:

1. `<meta name="theme-color">` tags (with media query support for light/dark modes)
2. Fixed/sticky header elements at the top of the page
3. Theme color from the web app manifest
4. Background color from the `<body>` or `<html>` element

The titlebar color updates dynamically as you navigate or when the page changes (scroll, resize, DOM mutations).

To disable this feature for a specific app, set `use_dynamic_titlebar_colors: false` in your .swai file.

## Best practices

- Use broad but safe allowed_urls patterns to cover all in-app routes you need.
- Add not_allowed_urls for domains you explicitly want opening in the external browser.
- Include login_urls for your provider’s auth endpoints to ensure multi-step auth can redirect properly.
- Keep app_id unique and stable; changing it will create a new data/session partition.

## Troubleshooting

- Links open in the external browser unexpectedly
  - Ensure both the currentUrl and navigationUrl are covered by allowed_urls (and not blocked by not_allowed_urls).
  - Check if you are on a login flow; add the provider’s login URLs to login_urls.

- New windows don’t open as separate app windows
  - Set allow_multi_window: true. If false, SWAI uses transient modal windows for window.open.

- App doesn’t retain login state across related apps (e.g., Word → Excel)
  - Set the same ecosystem for related apps to share sessions.

## Creating and placing the file

1) Create a file named something like myapp.swai with the YAML content above.
2) Place it in your chosen directory (e.g., testapps/ during development).
3) Launch SWAI and point it at your .swai file or include it in your registry as configured in your environment.

That’s it—your SWAI Web App will start with the configured URL and navigation rules.
