const { ipcMain, BrowserWindow } = require("electron");
const fs = require("fs");
const path = require("path");
const os = require("os");

class EcosystemManager {
    constructor(windowManager, instanceId) {
        this.windowManager = windowManager;
        this.instanceId = instanceId;
        this.ecosystemApps = new Map(); // Track apps in this ecosystem instance
        this.openWindows = new Map(); // Track open windows by app_id -> [windows]
        this.setupIPC();
        
        // Setup communication watcher with error handling
        try {
            this.setupCommunicationWatcher();
        } catch (error) {
            console.warn('Could not setup communication watcher:', error.message);
            console.log('Inter-process communication will not work, but single instances will still function');
        }
    }

    setupIPC() {
        try {
            // Listen for requests to open new apps in this ecosystem
            ipcMain.on('open-ecosystem-app', (event, webappData) => {
                this.handleNewAppRequest(webappData);
            });
        } catch (error) {
            console.warn('Could not setup IPC:', error.message);
        }
    }

    setupCommunicationWatcher() {
        try {
            // Create communication directory if it doesn't exist
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            if (!fs.existsSync(commDir)) {
                fs.mkdirSync(commDir, { recursive: true });
            }

            // Check for existing requests on startup
            this.checkForRequests();
            
            // Setup file watching with error handling
            const requestFile = path.join(commDir, `${this.instanceId}.request`);
            
            if (fs.existsSync(path.dirname(requestFile))) {
                // Use a more robust file watching approach
                this.watchInterval = setInterval(() => {
                    try {
                        this.checkForRequests();
                    } catch (error) {
                        console.warn('Error checking for requests:', error.message);
                    }
                }, 1000); // Check every second instead of using fs.watchFile
            }
        } catch (error) {
            console.warn('Could not setup communication watcher:', error.message);
        }
    }

    checkForRequests() {
        try {
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            const requestFile = path.join(commDir, `${this.instanceId}.request`);
            
            if (fs.existsSync(requestFile)) {
                const requestData = JSON.parse(fs.readFileSync(requestFile, 'utf-8'));
                console.log(`Received request to open: ${requestData.app_name}`);
                
                // Process the request
                this.handleNewAppRequest(requestData);
                
                // Clean up the request file
                fs.unlinkSync(requestFile);
            }
        } catch (error) {
            console.warn('Error processing app request:', error.message);
            // Try to clean up corrupted request file
            try {
                const commDir = path.join(os.tmpdir(), 'swai-comm');
                const requestFile = path.join(commDir, `${this.instanceId}.request`);
                if (fs.existsSync(requestFile)) {
                    fs.unlinkSync(requestFile);
                }
            } catch (cleanupError) {
                console.warn('Error cleaning up request file:', cleanupError.message);
            }
        }
    }

    async handleNewAppRequest(webappData) {
        try {
            const WebApp = require("../utils/WebApp");
            const webapp = Object.assign(new WebApp(), webappData);
            
            // Ensure SwaiRegistry is initialized if this is an ecosystem app
            if (webapp.ecosystem && this.windowManager.swaiRegistry) {
                if (!this.windowManager.swaiRegistry.initializationComplete) {
                    console.log(`Initializing SwaiRegistry for new ecosystem app: ${webapp.app_name}`);
                    await this.windowManager.swaiRegistry.initialize();
                }
            }
            
            // Check if this app is already running
            const existingWindows = this.openWindows.get(webapp.app_id) || [];
            
            if (existingWindows.length > 0) {
                // Same app is already running
                if (!webapp.allow_multi_window) {
                    // Bring existing window to front
                    console.log(`Bringing existing ${webapp.app_name} window to front`);
                    const firstWindow = existingWindows[0];
                    if (firstWindow && !firstWindow.isDestroyed()) {
                        firstWindow.show();
                        firstWindow.focus();
                        if (firstWindow.moveTop) {
                            firstWindow.moveTop();
                        }
                    }
                    return;
                } else {
                    // allow_multi_window is true, create another window
                    console.log(`Creating additional window for ${webapp.app_name}`);
                }
            } else {
                // Different app in same ecosystem, add it to our tracking
                console.log(`Opening new app in ecosystem: ${webapp.app_name}`);
                this.addApp(webapp);
            }
            
            // Create new window
            const newWindow = this.windowManager.createWindow(webapp);
            this.trackWindow(webapp.app_id, newWindow);
        } catch (error) {
            console.error('Error handling new app request:', error);
        }
    }

    addApp(webapp) {
        this.ecosystemApps.set(webapp.app_id, webapp);
        console.log(`Added ${webapp.app_name} to ecosystem instance`);
    }

    removeApp(appId) {
        this.ecosystemApps.delete(appId);
        console.log(`Removed app ${appId} from ecosystem instance`);
    }

    trackWindow(appId, window) {
        try {
            if (!this.openWindows.has(appId)) {
                this.openWindows.set(appId, []);
            }
            
            this.openWindows.get(appId).push(window);
            
            // Clean up when window is closed
            window.on('closed', () => {
                this.untrackWindow(appId, window);
            });
            
            console.log(`Now tracking ${this.openWindows.get(appId).length} windows for ${appId}`);
        } catch (error) {
            console.error('Error tracking window:', error);
        }
    }

    untrackWindow(appId, window) {
        try {
            if (this.openWindows.has(appId)) {
                const windows = this.openWindows.get(appId);
                const index = windows.indexOf(window);
                if (index > -1) {
                    windows.splice(index, 1);
                    
                    // Remove the app entirely if no windows left
                    if (windows.length === 0) {
                        this.openWindows.delete(appId);
                        this.removeApp(appId);
                    }
                }
            }
        } catch (error) {
            console.error('Error untracking window:', error);
        }
    }

    cleanup() {
        try {
            if (this.watchInterval) {
                clearInterval(this.watchInterval);
            }
        } catch (error) {
            console.warn('Error during cleanup:', error.message);
        }
    }

    getApps() {
        return Array.from(this.ecosystemApps.values());
    }

    getOpenWindows(appId = null) {
        if (appId) {
            return this.openWindows.get(appId) || [];
        }
        return this.openWindows;
    }
}

module.exports = EcosystemManager;