const {
    BaseWindow,
    BrowserWindow,
    WebContentsView,
    session,
    ipcMain,
} = require("electron");
const fs = require("fs");
const path = require("path");
const os = require("os");
const { parse_arg, openUrlInBrowser, wildCardMatchList } = require("../utils/swai_utils");
const Ecosystem = require("../utils/Ecosystem");

class WindowManager {
    constructor(debug_mode = false, ecosystemId = null, swaiRegistry = null) {
        this.windowId = 0;
        this.debug_mode = debug_mode;
        this.ecosystemId = ecosystemId;
        this.swaiRegistry = swaiRegistry;
        this.ecosystemManager = null; // Will be set by EcosystemManager
        this.dataPathsInitialized = false; // Track if app data paths have been set
    }
    
    setEcosystemManager(ecosystemManager) {
        this.ecosystemManager = ecosystemManager;
    }

    createWindow(webapp) {
        this.windowId += 1;
        let boundsTimeout = null;
        let webapp_ecosystem = null;

        // Use ecosystem ID for data path so apps in same ecosystem share data/cookies
        const dataId = this.ecosystemId || webapp.app_id;
        
        // Only set app data paths once per process (subsequent calls can cause issues)
        if (!this.dataPathsInitialized) {
            let data_path = path.join(os.homedir(), '.local', 'share', 'swai', 'app_data', dataId);
            const { app } = require("electron");
            app.setPath("userData", path.join(data_path, "userData"));
            app.setPath("appData", path.join(data_path, "appData"));
            app.setPath("sessionData", path.join(data_path, "sessionData"));
            app.setPath("cache", path.join(data_path, "cache"));
            app.setPath("logs", path.join(data_path, "logs"));
            app.setPath("crashDumps", path.join(data_path, "crashDumps"));
            this.dataPathsInitialized = true;
            console.log(`Initialized app data paths for: ${dataId}`);
        }

        // Use ecosystem ID for session partition so apps in same ecosystem share sessions/cookies
        let webAppSession = session.fromPartition(`persist:${dataId}`);

        let thisWindow = null;

        // For ecosystem apps, use internal title for DBus registration
        let windowTitle = webapp.app_name;
        let internalTitle = null;
        let currentDisplayTitle = webapp.app_name; // Track current display title to avoid redundant DBus calls
        
        if (webapp.ecosystem && this.swaiRegistry) {
            // Always generate internal title for ecosystem apps, even if DBus not available yet
            internalTitle = this.swaiRegistry.generateInternalTitle(webapp.app_id, this.windowId);
            windowTitle = internalTitle; // Set window title to internal title for DBus registration
        }

        thisWindow = new BaseWindow({
            width: 1200,
            height: 900,
            title: windowTitle,
            frame: false,
            show: true, // Explicitly show the window
        });
        
        console.log(`Created window ${this.windowId} for ${webapp.app_name}`);

        const set_title = (title) => {
            if (internalTitle && this.swaiRegistry && this.swaiRegistry.isAvailable()) {
                // For ecosystem apps, keep window title as internal title but update DBus display title
                thisWindow.title = internalTitle;
                
                // Only update DBus if the display title has actually changed
                if (title !== currentDisplayTitle) {
                    currentDisplayTitle = title;
                    this.swaiRegistry.updateTitle(internalTitle, title).catch(error => {
                        console.warn('Failed to update DBus title registration:', error.message);
                    });
                }
            } else {
                // For non-ecosystem apps, set title normally
                thisWindow.title = title;
            }
        };

        let titlebar = new WebContentsView({
            webPreferences: {
                contextIsolation: false,
                nodeIntegration: true,
                additionalArguments: [
                    `--windowId=${this.windowId}`,
                    `--webappName=${webapp.app_name}`,
                ],
            }
        });

        let mainWebContents = new WebContentsView({
            nodeIntegration: false,
            contextIsolation: true,
            webPreferences: {
                session: webAppSession
            }
        });

        titlebar.webContents.loadURL(
            `file://${path.join(__dirname, "../components/titlebar.html")}`,
        );
        thisWindow.removeMenu();
        thisWindow.contentView.addChildView(titlebar);
        thisWindow.contentView.addChildView(mainWebContents);

        let custom_main_url = parse_arg("--custom_main_url");
        if (custom_main_url) {
            webapp.main_url = custom_main_url;
            if (!wildCardMatchList(custom_main_url, webapp.allowed_urls)) {
                webapp.allowed_urls.push(custom_main_url);
            }
        }

        mainWebContents.webContents.loadURL(webapp.main_url);

        if (this.debug_mode) {
            titlebar.webContents.openDevTools();
            mainWebContents.webContents.openDevTools();
        }

        const updateBounds = () => {
            const { width, height } = thisWindow.getBounds();
            let titlebarHeight = this.debug_mode ? 300 : 36;
            titlebar.setBounds({ x: 0, y: 0, width, height: titlebarHeight });
            mainWebContents.setBounds({
                x: 0,
                y: titlebarHeight,
                width,
                height: height - titlebarHeight,
            });
        };

        updateBounds();

        thisWindow.on("resize", () => {
            updateBounds();
            if (boundsTimeout) {
                clearTimeout(boundsTimeout);
            }
            boundsTimeout = setTimeout(updateBounds, 1000);
        });

        this._setupMainWebContents(mainWebContents, thisWindow, webapp, set_title, webAppSession, titlebar);
        this._setupIpcHandlers(thisWindow, mainWebContents);
        this._setupContextMenu(mainWebContents);

        // Register the window with DBus for custom WM_CLASS handling (ecosystem apps only)
        if (webapp.ecosystem && this.swaiRegistry) {
            // Ensure SwaiRegistry is initialized before attempting registration
            // Don't await this - let it happen asynchronously so window creation isn't blocked
            this._registerWindowWithDelay(webapp, internalTitle, currentDisplayTitle, thisWindow).catch(err => {
                console.error('Window registration failed:', err);
            });
        }

        // Ensure window is visible
        thisWindow.show();
        console.log(`✓ Window ${this.windowId} for ${webapp.app_name} created and shown`);

        return thisWindow;
    }

    async _registerWindowWithDelay(webapp, internalTitle, currentDisplayTitle, thisWindow) {
        try {
            console.log(`Starting registration for window ${this.windowId} - ${webapp.app_name}`);
            
            // Ensure SwaiRegistry is initialized
            if (!this.swaiRegistry.initializationComplete) {
                console.log(`Waiting for SwaiRegistry initialization for ${webapp.app_name}...`);
                await this.swaiRegistry.initialize();
            }

            // Generate internal title if not already done
            if (!internalTitle) {
                internalTitle = this.swaiRegistry.generateInternalTitle(webapp.app_id, this.windowId);
                thisWindow.title = internalTitle;
                console.log(`Generated internal title: ${internalTitle}`);
            }

            if (internalTitle && this.swaiRegistry.isAvailable()) {
                const success = await this.swaiRegistry.registerTitle(internalTitle, currentDisplayTitle);
                if (success) {
                    console.log(`✓ Registered ${webapp.app_name} (window ${this.windowId}) with custom WM_CLASS: ${webapp.app_id}`);
                } else {
                    console.warn(`Failed to register ${webapp.app_name} with SwaiRegistry`);
                }
            } else {
                console.log(`DBus not available, skipping registration for ${webapp.app_name}`);
            }
        } catch (error) {
            console.error('Error during window registration:', error);
            // Don't let registration errors prevent the window from working
        }
    }

    _setupMainWebContents(mainWebContents, thisWindow, webapp, set_title, webAppSession, titlebar) {
        mainWebContents.webContents.on("dom-ready", () => {
            console.log("dom ready");

            if (thisWindow) {
                const cssPath = path.join(__dirname, "../../system_theme.css");
                fs.readFile(cssPath, "utf-8", (err, data) => {
                    if (err) {
                        console.error("Failed to read CSS file:", err);
                        return;
                    }
                    if (thisWindow) {
                        mainWebContents.webContents.insertCSS(data).catch((error) => {
                            console.error("Failed to inject CSS:", error);
                        });
                    }
                });
            }

            mainWebContents.webContents.setWindowOpenHandler((details) => {
                console.log("Attempted new window:", details.url);
                console.log("Current URL:", mainWebContents.webContents.getURL());
                console.log("Multi-window allowed:", webapp.allow_multi_window);
                
                if (!this._get_navigation_allowed(
                    mainWebContents.webContents.getURL(),
                    details.url,
                    webapp
                )) {
                    console.log("Navigation not allowed, opening in browser");
                    openUrlInBrowser(details.url);
                    return { action: "deny" };
                }
                if (!webapp.allow_multi_window) {
                    console.log("Multi-window not allowed, creating transient window");
                    this.createTransientWindow(details.url, webAppSession, thisWindow);
                    return { action: "deny" };
                }
                if (wildCardMatchList(details.url, webapp.allowed_urls)) {
                    console.log("URL matches allowed list, creating new window");
                    try {
                        let new_window_app = webapp.clone();
                        new_window_app.main_url = details.url;
                        const newWindow = this.createWindow(new_window_app);
                        
                        // Track the new window in the ecosystem manager
                        if (this.ecosystemManager) {
                            this.ecosystemManager.trackWindow(new_window_app.app_id, newWindow);
                            console.log("New window tracked in ecosystem manager");
                        }
                        
                        console.log("New window created successfully");
                        return { action: "deny" };
                    } catch (error) {
                        console.error("Error creating new window:", error);
                        return { action: "deny" };
                    }
                } else {
                    console.log("URL does not match allowed list, creating transient window");
                    this.createTransientWindow(details.url, webAppSession, thisWindow);
                    return { action: "deny" };
                }
            });

            this._setupNavigationHandlers(mainWebContents, webapp, set_title, titlebar);
            this._setupKeyboardShortcuts(mainWebContents);
        });
    }

    _setupNavigationHandlers(mainWebContents, webapp, set_title, titlebar) {
        mainWebContents.webContents.on("will-navigate", (event, navigationUrl) => {
            if (!this._get_navigation_allowed(
                mainWebContents.webContents.getURL(),
                navigationUrl,
                webapp
            )) {
                event.preventDefault();
                openUrlInBrowser(navigationUrl);
            }
        });

        mainWebContents.webContents.on("did-navigate", (event, navigationUrl) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
        });

        mainWebContents.webContents.on("did-navigate-in-page", (event, navigationUrl) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
        });

        mainWebContents.webContents.on("page-title-updated", (event, title) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
        });
    }

    _sendUrlChanged(mainWebContents, titlebar) {
        if (titlebar) {
            titlebar.webContents.send(
                `url-changed-${this.windowId}`,
                mainWebContents.webContents.getTitle(),
                mainWebContents.webContents.navigationHistory.canGoBack(),
                mainWebContents.webContents.navigationHistory.canGoForward(),
            );
        }
    }

    _setupKeyboardShortcuts(mainWebContents) {
        mainWebContents.webContents.on("before-input-event", (event, input) => {
            if (input.control) {
                if (input.key === "I" && input.control && input.shift) {
                    mainWebContents.webContents.toggleDevTools();
                    event.preventDefault();
                } else if (input.key === "+") {
                    mainWebContents.webContents.setZoomFactor(
                        mainWebContents.webContents.getZoomFactor() + 0.1,
                    );
                    event.preventDefault();
                } else if (input.key === "-") {
                    mainWebContents.webContents.setZoomFactor(
                        mainWebContents.webContents.getZoomFactor() - 0.1,
                    );
                    event.preventDefault();
                } else if (input.key === "0") {
                    mainWebContents.webContents.setZoomFactor(1.0);
                    event.preventDefault();
                } else if (input.key == "r") {
                    mainWebContents.webContents.reload();
                    event.preventDefault();
                }
            }
        });
    }

    _setupIpcHandlers(thisWindow, mainWebContents) {
        ipcMain.on(`back-${this.windowId}`, () => {
            if (thisWindow) {
                mainWebContents.webContents.navigationHistory.goBack();
            }
        });

        ipcMain.on(`forward-${this.windowId}`, () => {
            if (thisWindow) {
                mainWebContents.webContents.navigationHistory.goForward();
            }
        });

        ipcMain.on(`close-${this.windowId}`, () => {
            if (thisWindow) {
                thisWindow.close();
            }
        });

        ipcMain.on(`minimize-${this.windowId}`, () => {
            if (thisWindow) {
                thisWindow.minimize();
            }
        });

        ipcMain.on(`maximize-${this.windowId}`, () => {
            if (thisWindow) {
                if (thisWindow.isMaximized()) {
                    thisWindow.unmaximize();
                } else {
                    thisWindow.maximize();
                }
            }
        });
    }

    async _setupContextMenu(mainWebContents) {
        try {
            const contextMenu = await import("electron-context-menu");
            const dispose = contextMenu.default({
                window: mainWebContents.webContents,
                showInspectElement: false,
            });
        } catch (error) {
            console.error("Failed to setup context menu:", error);
        }
    }

    _get_navigation_allowed(currentUrl, navigationUrl, webapp) {
        if (wildCardMatchList(navigationUrl, webapp.not_allowed_urls)) {
            console.log("Opening browser window");
            return false;
        }
        if (wildCardMatchList(currentUrl, webapp.login_urls)) {
            console.log("Bypassed Browser Window1");
            return true;
        }
        if (!wildCardMatchList(currentUrl, webapp.allowed_urls)) {
            console.log("Bypassed Browser Window2");
            return true;
        }
        if (wildCardMatchList(navigationUrl, webapp.allowed_urls)) {
            return true;
        }
        return false;
    }

    createTransientWindow(url, session, parent) {
        let transientWindow = new BrowserWindow({
            width: 800,
            height: 600,
            parent: parent,
            modal: true,
            title: "SWAI Web App",
            webPreferences: {
                session: session,
            },
        });
        transientWindow.loadURL(url);
        transientWindow.once("ready-to-show", () => {
            transientWindow.show();
        });
        transientWindow.removeMenu();
    }
}

module.exports = WindowManager; 