const {
    app,
    BrowserWindow,
    WebContentsView,
    BaseWindow,
    session,
    ipcMain,
    dialog,
} = require("electron");
const WebApp = require("./src/utils/WebApp");
const {
    parse_arg,
    openUrlInBrowser,
    wildCardMatchList,
} = require("./src/utils/swai_utils");
const path = require("path");
const fs = require("fs");
const Ecosystem = require("./src/utils/Ecosystem");
const os = require("os");

// Import new manager classes
const LockFileManager = require("./src/managers/LockFileManager");
const WindowManager = require("./src/managers/WindowManager");
const EcosystemLauncher = require("./src/managers/EcosystemLauncher");
const EcosystemManager = require("./src/managers/EcosystemManager");
const SwaiRegistry = require("./src/managers/SwaiRegistry");

// Check for the debug mode argument
let debug_mode =
    parse_arg("--debug_mode") != null && parse_arg("--debug_mode") === "true";

let windowManager = null;
let ecosystemManager = null;
let swaiRegistry = null;

app.on("window-all-closed", () => {
    app.quit();
});

app.on("ready", async () => {
    try {
        const filePath = process.argv[2];
        if (!filePath) {
            console.error("No file path provided");
            app.quit();
            return;
        }
        if (!fs.existsSync(filePath)) {
            console.error(`File doesn't exist: ${filePath}`);
            app.quit();
            return;
        }
        const webapp = WebApp.from_yaml_file(filePath);

        // Use ecosystem-based launching instead of app_id-based
        if (!EcosystemLauncher.launch(webapp)) {
            // EcosystemLauncher.launch returns false if the request was sent to an existing instance
            // In this case, we should exit this process since the app will be opened in the existing instance
            console.log("App request sent to existing ecosystem instance, exiting...");
            app.quit();
            return;
        }

        const ecosystemId = EcosystemLauncher.getEcosystemId(webapp);
        
        app.commandLine.appendSwitch("ozone-platform-hint", "auto");
        
        // Set WM_CLASS to always be io.stillhq.swai.{app_id} format
        app.setName(`io.stillhq.swai.${webapp.app_id}`);

        // Initialize components
        swaiRegistry = new SwaiRegistry();
        
        // Initialize SwaiRegistry asynchronously before creating windows
        console.log(`Initializing SwaiRegistry for app: ${webapp.app_name}...`);
        await swaiRegistry.initialize();
        console.log('SwaiRegistry initialization complete');
        
        windowManager = new WindowManager(debug_mode, ecosystemId, swaiRegistry);
        ecosystemManager = new EcosystemManager(windowManager, ecosystemId);
        
        // Set the ecosystem manager reference in window manager
        windowManager.setEcosystemManager(ecosystemManager);

        // Create the main window
        const mainWindow = windowManager.createWindow(webapp);
        
        // Track this app in the ecosystem
        ecosystemManager.addApp(webapp);
        ecosystemManager.trackWindow(webapp.app_id, mainWindow);

        // Set up cleanup on app quit
        app.on('before-quit', async () => {
            if (swaiRegistry) {
                await swaiRegistry.cleanup();
            }
        });

    } catch (error) {
        console.error("Error during app startup:", error);
        app.quit();
    }
});