#!/bin/bash

# StillCenter System-Wide Test Installation Script
# Based on stillcenter.spec installation structure
# This should be used for debugging purpsoes only,
# this was vibe-coded with Cursor/Claude4.

set -e  # Exit on any error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}Error: This script must be run as root for system-wide installation${NC}"
   echo "Usage: sudo ./test_install.sh"
   exit 1
fi

echo -e "${GREEN}StillCenter System-Wide Test Installation${NC}"
echo "========================================"

# Get Python site-packages directory (system-wide)
PYTHON_SITELIB=$(python3 -c "import sys; print(sys.path[-1])")
if [[ "$PYTHON_SITELIB" == *"/usr/local/"* ]]; then
    # Find alternative system site-packages
    PYTHON_SITELIB=$(python3 -c "import site; print([p for p in site.getsitepackages() if not p.startswith('/usr/local')][0])")
fi

# Define installation paths (avoiding /usr/local)
BINDIR="/usr/bin"
DATADIR="/usr/share"
ICONDIR="$DATADIR/icons/hicolor/scalable/apps"
APPDIR="$DATADIR/applications"
STILLCENTER_DIR="$PYTHON_SITELIB/stillCenter"

echo -e "${YELLOW}Installation paths:${NC}"
echo "  Python modules: $STILLCENTER_DIR"
echo "  Binary: $BINDIR/stillcenter"
echo "  Desktop file: $APPDIR/io.stillhq.stillCenter.desktop"
echo "  Icon: $ICONDIR/io.stillhq.stillCenter.svg"
echo ""

# Check dependencies
echo -e "${YELLOW}Checking dependencies...${NC}"
if ! command -v python3 &> /dev/null; then
    echo -e "${RED}Error: python3 is required but not installed${NC}"
    exit 1
fi

if ! command -v glib-compile-resources &> /dev/null; then
    echo -e "${RED}Error: glib-compile-resources is required (install glib2-devel)${NC}"
    exit 1
fi

# Check if sam and sadb are available (as per spec requirements)
for dep in sam sadb; do
    if ! command -v $dep &> /dev/null; then
        echo -e "${YELLOW}Warning: $dep is not installed (required dependency)${NC}"
    fi
done

echo -e "${GREEN}Dependencies check passed${NC}"
echo ""

# Create directories
echo -e "${YELLOW}Creating installation directories...${NC}"
mkdir -p "$STILLCENTER_DIR"
mkdir -p "$ICONDIR"
mkdir -p "$APPDIR"

# Install Python modules
echo -e "${YELLOW}Installing Python modules...${NC}"
cp -a . "$STILLCENTER_DIR/"

# Remove development files from installation
rm -f "$STILLCENTER_DIR/.gitignore" 2>/dev/null || true
rm -rf "$STILLCENTER_DIR/.git" 2>/dev/null || true
rm -rf "$STILLCENTER_DIR/.vscode" 2>/dev/null || true
rm -rf "$STILLCENTER_DIR/.idea" 2>/dev/null || true
rm -rf "$STILLCENTER_DIR/__pycache__" 2>/dev/null || true
rm -f "$STILLCENTER_DIR/test_install.sh" 2>/dev/null || true
rm -f "$STILLCENTER_DIR/stillcenter.spec" 2>/dev/null || true

# Install binary
echo -e "${YELLOW}Installing binary...${NC}"
install -m 0755 __main__.py "$BINDIR/stillcenter"

# Install desktop file
echo -e "${YELLOW}Installing desktop file...${NC}"
if [[ -f "data/io.stillhq.stillCenter.desktop" ]]; then
    install -m 0644 data/io.stillhq.stillCenter.desktop "$APPDIR/io.stillhq.stillCenter.desktop"
else
    echo -e "${RED}Error: Desktop file not found at data/io.stillhq.stillCenter.desktop${NC}"
    exit 1
fi

# Install icon
echo -e "${YELLOW}Installing icon...${NC}"
if [[ -f "data/io.stillhq.stillCenter.svg" ]]; then
    install -m 0644 data/io.stillhq.stillCenter.svg "$ICONDIR/io.stillhq.stillCenter.svg"
else
    echo -e "${RED}Error: Icon file not found at data/io.stillhq.stillCenter.svg${NC}"
    exit 1
fi

# Compile GResource
echo -e "${YELLOW}Compiling GResource...${NC}"
if [[ -f "data/io.stillhq.stillCenter.gresource.xml" ]]; then
    glib-compile-resources --sourcedir data --target "$STILLCENTER_DIR/data/io.stillhq.stillCenter.gresource" data/io.stillhq.stillCenter.gresource.xml
else
    echo -e "${RED}Error: GResource XML file not found${NC}"
    exit 1
fi

# Update desktop database
echo -e "${YELLOW}Updating desktop database...${NC}"
if command -v update-desktop-database &> /dev/null; then
    update-desktop-database "$APPDIR" 2>/dev/null || true
fi

# Update icon cache
echo -e "${YELLOW}Updating icon cache...${NC}"
if command -v gtk-update-icon-cache &> /dev/null; then
    gtk-update-icon-cache -f -t "$DATADIR/icons/hicolor" 2>/dev/null || true
fi

echo ""
echo -e "${GREEN}✓ StillCenter has been installed system-wide successfully!${NC}"
echo ""
echo -e "${YELLOW}Installation summary:${NC}"
echo "  • Binary: $BINDIR/stillcenter"
echo "  • Python modules: $STILLCENTER_DIR"
echo "  • Desktop file: $APPDIR/io.stillhq.stillCenter.desktop"
echo "  • Icon: $ICONDIR/io.stillhq.stillCenter.svg"
echo ""
echo -e "${YELLOW}Usage:${NC}"
echo "  • Run from terminal: stillcenter"
echo "  • Run from applications menu: Search for 'StillCenter'"
echo ""
echo -e "${YELLOW}To uninstall:${NC}"
echo "  sudo rm -rf $STILLCENTER_DIR"
echo "  sudo rm -f $BINDIR/stillcenter"
echo "  sudo rm -f $APPDIR/io.stillhq.stillCenter.desktop"
echo "  sudo rm -f $ICONDIR/io.stillhq.stillCenter.svg"
echo "" 