const fs = require("fs");
const path = require("path");
const os = require("os");
const { spawn } = require("child_process");

class EcosystemLauncher {
    /**
     * Determines the ecosystem ID for a webapp
     * If the webapp has an ecosystem, use that; otherwise use app_id
     */
    static getEcosystemId(webapp) {
        return webapp.ecosystem || webapp.app_id;
    }

    /**
     * Checks if an ecosystem instance is already running
     */
    static isEcosystemRunning(ecosystemId) {
        try {
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            const lockFile = path.join(commDir, `${ecosystemId}.lock`);
            
            if (fs.existsSync(lockFile)) {
                const pid = parseInt(fs.readFileSync(lockFile, 'utf-8'));
                return this.isProcessRunning(pid);
            }
            return false;
        } catch (error) {
            console.warn('Error checking ecosystem status:', error.message);
            return false;
        }
    }

    /**
     * Checks if a process is running
     */
    static isProcessRunning(pid) {
        try {
            process.kill(pid, 0);
            return true;
        } catch (e) {
            return false;
        }
    }

    /**
     * Creates a lock file for the ecosystem instance
     */
    static createEcosystemLock(ecosystemId) {
        try {
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            if (!fs.existsSync(commDir)) {
                fs.mkdirSync(commDir, { recursive: true });
            }
            
            const lockFile = path.join(commDir, `${ecosystemId}.lock`);
            fs.writeFileSync(lockFile, process.pid.toString());
            
            // Clean up lock file on exit
            process.on('exit', () => {
                try {
                    if (fs.existsSync(lockFile)) {
                        fs.unlinkSync(lockFile);
                    }
                } catch (error) {
                    console.warn('Error cleaning up lock file:', error.message);
                }
            });
            
            return true;
        } catch (error) {
            console.error('Error creating ecosystem lock:', error);
            return false;
        }
    }

    /**
     * Removes the lock file for the ecosystem instance
     */
    static removeEcosystemLock(ecosystemId) {
        try {
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            const lockFile = path.join(commDir, `${ecosystemId}.lock`);
            if (fs.existsSync(lockFile)) {
                fs.unlinkSync(lockFile);
            }
        } catch (error) {
            console.warn('Error removing ecosystem lock:', error.message);
        }
    }

    /**
     * Sends a request to an existing ecosystem instance to open a new app
     */
    static requestAppInEcosystem(ecosystemId, webapp) {
        try {
            const commDir = path.join(os.tmpdir(), 'swai-comm');
            const requestFile = path.join(commDir, `${ecosystemId}.request`);
            
            // Write request data
            const requestData = {
                app_id: webapp.app_id,
                app_name: webapp.app_name,
                main_url: webapp.main_url,
                allowed_urls: webapp.allowed_urls,
                not_allowed_urls: webapp.not_allowed_urls,
                allow_multi_window: webapp.allow_multi_window,
                ecosystem: webapp.ecosystem,
                login_urls: webapp.login_urls,
                swai_file_path: webapp.swai_file_path
            };
            
            fs.writeFileSync(requestFile, JSON.stringify(requestData));
            console.log(`Requested ${webapp.app_name} to be opened in existing ${ecosystemId} instance`);
            return true;
        } catch (error) {
            console.error('Error sending app request:', error);
            return false;
        }
    }

    /**
     * Launches a new swai instance for the given webapp
     * Returns true if this should be the main instance, false if request was sent to existing instance
     */
    static launch(webapp) {
        const ecosystemId = this.getEcosystemId(webapp);
        
        console.log(`Ecosystem ID for ${webapp.app_name}: ${ecosystemId}`);
        
        // Check if ecosystem instance is already running
        if (this.isEcosystemRunning(ecosystemId)) {
            console.log(`Ecosystem ${ecosystemId} is already running, sending request...`);
            
            if (this.requestAppInEcosystem(ecosystemId, webapp)) {
                // Request sent successfully, this process should exit
                return false;
            } else {
                console.warn('Failed to send request to existing instance, launching new instance');
            }
        }
        
        // Create lock file for this ecosystem instance
        if (!this.createEcosystemLock(ecosystemId)) {
            console.error('Failed to create ecosystem lock, exiting');
            return false;
        }
        
        console.log(`Starting new ecosystem instance for: ${ecosystemId}`);
        return true;
    }
}

module.exports = EcosystemLauncher; 