/*
 * Copyright 2003-2023 The IdeaVim authors
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE.txt file or at
 * https://opensource.org/licenses/MIT.
 */

package com.maddyhome.idea.vim.config

import org.jdom.Element
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty

/**
 * @author Alex Plate
 */
internal class VimState {
  var isIdeaJoinNotified by StateProperty("idea-join")
  var isIdeaPutNotified by StateProperty("idea-put")
  var wasSubscribedToEAPAutomatically by StateProperty("was-automatically-subscribed-to-eap")
  var firstIdeaVimVersion: String? by StringProperty("first-ideavim-version", null)

  fun readData(element: Element) {
    val notifications = element.getChild("notifications")
    map.keys.forEach { name ->
      notifications?.getChild(name)?.getAttributeValue("enabled")?.let {
        map[name] = it.toBoolean()
      }
    }
    stringMap.keys.forEach { name ->
      notifications?.getChild(name)?.getAttributeValue("value")?.let {
        stringMap[name] = it.decode
      }
    }
  }

  fun saveData(element: Element) {
    val notifications = Element("notifications")
    element.addContent(notifications)

    map.forEach { (name, value) ->
      val child = Element(name)
      child.setAttribute("enabled", value.toString())
      notifications.addContent(child)
    }
    stringMap.forEach { (name, value) ->
      val child = Element(name)
      child.setAttribute("value", value.encode)
      notifications.addContent(child)
    }
  }
}

private val String?.encode: String get() = this ?: NULL_VALUE
private val String?.decode: String? get() = if (this == NULL_VALUE) null else this

// Settings cannot store null values
private const val NULL_VALUE = "__NULL_VALUE_CONST__"

private val map by lazy { mutableMapOf<String, Boolean>() }
private val stringMap by lazy { mutableMapOf<String, String?>() }

private class StateProperty(val xmlName: String) : ReadWriteProperty<VimState, Boolean> {

  init {
    map[xmlName] = false
  }

  override fun getValue(thisRef: VimState, property: KProperty<*>): Boolean = map.getOrPut(xmlName) { false }

  override fun setValue(thisRef: VimState, property: KProperty<*>, value: Boolean) {
    map[xmlName] = value
  }
}

private class StringProperty(val propertyName: String, val defaultValue: String?) :
  ReadWriteProperty<VimState, String?> {

  init {
    stringMap[propertyName] = defaultValue
  }

  override fun getValue(thisRef: VimState, property: KProperty<*>): String? {
    return stringMap.getOrPut(propertyName) { defaultValue }
  }

  override fun setValue(thisRef: VimState, property: KProperty<*>, value: String?) {
    stringMap[propertyName] = value
  }
}
