#!/usr/bin/env python
#
# Station driver for Weather Radio https://github.com/indilib/indi-3rdparty/blob/master/indi-duino/doc/Readme-WeatherRadio.md
# Based on arduinoPWS driver https://github.com/rkaczorek/arduino-pws
#
# For minimal changes to weatherradio I'm trying to support c v and w commands
# v and c commands are saved on init as a static website. Rather than generating
# w in the driver, instead it's an output generated by weewx which potentially
# allows it to be decorated in the long run by extra information from extensions
# and calculated data.
#
# I've implemented this using using a custom skin report which updates every ARCHIVE
# record
#
# Even better would be to have a version of the weatherradio indi driver that pulls
# its data real-time from weewx every LOOP. This could be implemented as a custom weewx service
# writing out JSON, or by altering the indi driver to read from the output of the generic
# JSON format produced by the service https://github.com/chaunceygardiner/weewx-loopdata
#

from __future__ import with_statement
import serial
import syslog
import time
import json

import weewx.drivers
import weeutil.weeutil

DRIVER_NAME = 'weatherradio'
DRIVER_VERSION = '0.1'

# Serial port to the arduino weather station (I use XBee modules for easy serial connection with my PI)
DEFAULT_PORT = '/dev/serial0'
DEBUG_SERIAL = 0

# Directory for the website needed by the weatherradio indi driver
DEFAULT_WEBROOT = '/var/www/weatherradio'

def loader(config_dict, _):
    return weatherradio(**config_dict[DRIVER_NAME])

def confeditor_loader():
    return weatherradioConfEditor()

def logmsg(level, msg):
    syslog.syslog(level, 'weatherradio: %s' % msg)

def logdbg(msg):
    logmsg(syslog.LOG_DEBUG, msg)

def loginf(msg):
    logmsg(syslog.LOG_INFO, msg)

def logerr(msg):
    logmsg(syslog.LOG_ERR, msg)


class weatherradio(weewx.drivers.AbstractDevice):
    def __init__(self, **stn_dict):
        self.model = stn_dict.get('model', 'weatherradio')
        self.port = stn_dict.get('port', DEFAULT_PORT)
        self.webroot = stn_dict.get('webroot', DEFAULT_WEBROOT)
        self.max_tries = int(stn_dict.get('max_tries', 10))
        self.retry_wait = int(stn_dict.get('retry_wait', 10))

        global DEBUG_SERIAL
        DEBUG_SERIAL = int(stn_dict.get('debug_serial', DEBUG_SERIAL))

        loginf('driver version is %s' % DRIVER_VERSION)
        loginf('using serial port %s' % self.port)
        self.station = Station(self.port)
        self.station.open()
        self.initweb()

    def closePort(self):
        if self.station is not None:
            self.station.close()
            self.station = None

    @property
    def hardware_name(self):
        return self.model

    def initweb(self):
        buf = self.station.get_version()
        # Write to webroot
        f = open(self.webroot + '/v', "w")
        f.write(buf.decode('utf-8'))
        f.close()

        buf = self.station.get_config()
        # Write to webroot
        f = open(self.webroot + '/c', "w")
        f.write(buf.decode('utf-8'))
        f.close()

    def genLoopPackets(self):
        while True:
            packet = {'dateTime': int(time.time() + 0.5),
                      'usUnits': weewx.METRIC}
            readings = self.station.get_readings_with_retry(self.max_tries, self.retry_wait)
            data = Station.parse_readings(readings)
            packet.update(data)
            yield packet

            # Write to webroot
            # Now doing this as an output of weewx rather than in the driver - see above
#            f = open(self.webroot + '/w', "w")
#            f.write(readings.decode('utf-8'))
#            f.close()

            time.sleep(10)

class Station(object):
    def __init__(self, port):
        self.port = port
        self.baudrate = 9600
        self.timeout = 30
        self.serial_port = None

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, _, value, traceback):
        self.close()

    def open(self):
        logdbg("open serial port %s" % self.port)
        self.serial_port = serial.Serial(self.port, self.baudrate,
                                         timeout=self.timeout)

    def close(self):
        if self.serial_port is not None:
            logdbg("close serial port %s" % self.port)
            self.serial_port.close()
            self.serial_port = None

    def get_readings(self):
        self.serial_port.write("w\n".encode())
        buf = self.serial_port.readline()
        buf = buf.strip()
        if DEBUG_SERIAL:
            logdbg("station said: %s" % buf)
        return buf

    def get_version(self):
        self.serial_port.write("v\n".encode())
        buf = self.serial_port.readline()
        buf = buf.strip()
        if DEBUG_SERIAL:
            logdbg("station said: %s" % buf)
        return buf

    def get_config(self):
        self.serial_port.write("c\n".encode())
        buf = self.serial_port.readline()
        buf = buf.strip()
        if DEBUG_SERIAL:
            logdbg("station said: %s" % buf)
        return buf

    def validate_string(self, buf):
        try:
            json.loads(buf)
            return buf
        except:
            raise weewx.WeeWxIOError("Invalid JSON")

    def get_readings_with_retry(self, max_tries=5, retry_wait=10):
        for ntries in range(0, max_tries):
            try:
                buf = self.get_readings()
                self.validate_string(buf)
                return buf
            except (serial.serialutil.SerialException, weewx.WeeWxIOError) as e:
                loginf("Failed attempt %d of %d to get readings: %s" %
                       (ntries + 1, max_tries, e))
                time.sleep(retry_wait)
        else:
            msg = "Max retries (%d) exceeded for readings" % max_tries
            logerr(msg)
            raise weewx.RetriesExceeded(msg)

    @staticmethod
    def parse_readings(raw):
        data = dict()
        readings = json.loads(raw)

        data['outHumidity'] = readings['BME280']['Hum']
        data['outTemp'] = readings['BME280']['Temp']
        data['pressure'] = readings['BME280']['Pres']
        data['illuminance'] = readings['TSL2591']['Lux']
        data['extraTemp1'] = readings['MLX90614']['T amb']
        data['extraTemp2'] = readings['MLX90614']['T obj']

        return data

class weatherradioConfEditor(weewx.drivers.AbstractConfEditor):
    @property
    def default_stanza(self):
        return """
[weatherradio]
    # This section is for Weather Radio

    # Serial port such as /dev/rfcomm0 (default) or /dev/serial0
    port = /dev/serial0

    # webroot for weatherradio indi driver website
    webroot = /var/www/weatherradio

    # The driver to use:
    driver = user.weatherradio
"""

    def prompt_for_settings(self):
        print("Specify the serial port on which the station is connected, for")
        print("example /dev/rfcomm0 or /dev/serial0")
        port = self._prompt('port', '/dev/serial0')
        return {'port': port}


# define a main entry point for basic testing of the station without weewx
# engine and service overhead.  invoke this as follows:
# PYTHONPATH=/usr/share/weewx python3 bin/user/weatherradio.py

if __name__ == '__main__':
    import optparse

    usage = """%prog [options] [--help]"""

    syslog.openlog('weatherradio', syslog.LOG_PID | syslog.LOG_CONS)
    syslog.setlogmask(syslog.LOG_UPTO(syslog.LOG_DEBUG))
    parser = optparse.OptionParser(usage=usage)
    parser.add_option('--version', dest='version', action='store_true',
                      help='display driver version')
    parser.add_option('--port', dest='port', metavar='PORT',
                      help='serial port to which the station is connected',
                      default=DEFAULT_PORT)
    (options, args) = parser.parse_args()

    if options.version:
        print("weatherradio driver version %s" % DRIVER_VERSION)
        exit(0)

    with Station(options.port) as s:
        while True:
            print(time.time(), Station.parse_readings(s.get_readings()))
            time.sleep(5)
