/*
 Raspberry Pi High Quality Camera CCD Driver for Indi.
 Copyright (C) 2020 Lars Berntzon (lars.berntzon@cecilia-data.se).
 All rights reserved.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdexcept>
#include "jpegpipeline.h"
#include "inditest.h"

void JpegPipeline::reset()
{
    state = State::WANT_FF;
    s1 = 0;
    s2 = 0;
    skip_bytes = 0;
    entropy_data_follows = false;
    current_type = 0;
}

/**
 * @brief RawStreamReceiver::accept_byte Accepts next byte of input and sets state accordingly.
 * @param byte Next byte to handle.
 * @return
 */
 void JpegPipeline::data_received(uint8_t *data,  uint32_t length)
{
     uint8_t byte;
     while(length > 0)
     {
        byte = *data;
        switch(state)
        {
        case State::END_OF_JPEG:
            // Fast lane, forward all data upwards from here.
            forward(data, length);
            return;

        case State::INVALID:
            throw Exception("State invalid");
            break;

        case State::WANT_FF:
            if (byte != 0xFF) {
                throw Exception("Expected 0xFF");
            }
            state = State::WANT_TYPE;
            break;

        case State::WANT_S1:
            s1 = byte;
            state = State::WANT_S2;
            break;

        case State::WANT_S2:
            s2 = byte;
            state = State::SKIP_BYTES;
            skip_bytes = (s1 << 8) + s2 - 2;  // -2 since we already read S1 S2
            break;

        case State::SKIP_BYTES:
	    if(skip_bytes <= length)
	    {
		length -= skip_bytes;
		data += skip_bytes;
		skip_bytes = 0;
	    }
	    else
	    {
		length = 0;
		data += length;
		skip_bytes -= length;
	    }
            if (skip_bytes == 0) {
                if (entropy_data_follows) {
                    state = State::WANT_ENTROPY_DATA;
                }
                else {
                    state = State::WANT_FF;
                }
            }
            continue;

        case State::WANT_ENTROPY_DATA:
            if (byte == 0xFF) {
                state = State::ENTROPY_GOT_FF;
            }
            break;

        case State::ENTROPY_GOT_FF:
            if (byte == 0) {
                // Just an escaped 0
                state = State::WANT_ENTROPY_DATA;
                break;
            }
            else if (byte == 0xFF) {
                // Padding
                state = State::ENTROPY_GOT_FF;
                break;
            }
            // If not FF00 and FFFF then we got a real segment type now.
            state = State::WANT_TYPE;
            entropy_data_follows = false;

            // FALL THROUGH

        case State::WANT_TYPE:
            current_type = byte;
            switch(byte)
            {
            case 0xd8: // SOI (Start of image)
                state = State::WANT_FF;
                break;

            case 0xd9: // EOI (End of image)
                LOG_TEST("finished jpeg processing"); 
                state = State::END_OF_JPEG;
                break;

            case 0xda: // SOS (Start of stream)
            case 0xc0: // Baseline DCT
            case 0xc4: // Huffman Table
                entropy_data_follows = true;
                // Above sections will have entropy data following.
                // Fall through
            case 0xdb: // Quantization Table
            case 0xe0: // JFIF APP0
            case 0xe1: // JFIF APP0
                state = State::WANT_S1;
                break;

            default:
                throw Exception("Unknown JPEG segment type.");
                return;
            }
            break;
        }
	data++;
	length--;
     }
}
